import datetime
import json

from django.conf import settings
from django.contrib.auth.decorators import permission_required
from django.contrib.auth.models import AnonymousUser
from django.core.exceptions import PermissionDenied, BadRequest
from django.core.paginator import Paginator
from django.http import HttpRequest, HttpResponse, Http404
from django.shortcuts import redirect, render
from django.utils.text import slugify

from ..models import (
    Category,
    Note,
    Sheet,
    Map,
    ProgressCounter,
    ProgressLog,
    Calendar,
    PushSubscription,
    Project,
    MoodLog)

from .utils import (
    ConflictError,
    find_user_object,
    pretty_paginator,
    save_document_core,
    get_or_create_settings,
    toggle_object_attribute,
    view_documents_single,
    view_documents_mixed,
    get_pending_mood_logs)


# GENERAL ######################################################################


def view_landing(request: HttpRequest) -> HttpResponse:
    if request.user.is_authenticated:
        return redirect("orgapy:projects")
    return redirect("orgapy:about")


def view_about(request: HttpRequest) -> HttpResponse:
    """View for the homepage, describing the application for a new user."""
    return render(request, "orgapy/about.html", {})


# PROJECTS #####################################################################


@permission_required("orgapy.view_project")
def view_projects(request: HttpRequest) -> HttpResponse:
    if isinstance(request.user, AnonymousUser):
        raise PermissionDenied()
    settings = get_or_create_settings(request.user)
    pending_mood_logs = get_pending_mood_logs(request.user, settings.mood_log_hours)
    return render(request, "orgapy/projects.html", {
        "settings": settings,
        "pending_mood_logs": pending_mood_logs,
        "active": "projects",
    })


@permission_required("orgapy.view_project")
def view_projects_all(request: HttpRequest) -> HttpResponse:
    note_filter = request.GET.get("note")
    status_filter = request.GET.get("status")
    query = Project.objects.filter(user=request.user)
    if note_filter is not None:
        try:
            query = query.filter(note__id=int(note_filter))
        except:
            raise BadRequest()
    if status_filter is not None:
        query = query.filter(status=status_filter)
    query = query.order_by("-date_modification")
    page_size = 24
    paginator = Paginator(query, page_size)
    page = request.GET.get("page")
    projects = paginator.get_page(page)
    return render(request, "orgapy/projects_all.html", {
        "projects": projects,
        "paginator": pretty_paginator(projects),
        "active": "projects",
    })


@permission_required("orgapy.view_project")
def view_project(request: HttpRequest, object_id: str) -> HttpResponse:
    if isinstance(request.user, AnonymousUser):
        raise PermissionDenied()
    project = find_user_object(Project, "id", object_id, request.user)
    return render(request, "orgapy/project.html", {
        "project": project,
        "active": "projects",
    })


@permission_required("orgapy.delete_project")
def view_delete_project(request: HttpRequest, object_id: str) -> HttpResponse:
    project = find_user_object(Project, "id", object_id, request.user)
    project.delete()
    if "next" in request.GET:
        return redirect(request.GET["next"])
    return redirect("orgapy:projects")


# OBJECTS AND CATEGORIES #######################################################


@permission_required("orgapy.view_note")
@permission_required("orgapy.view_sheet")
@permission_required("orgapy.view_map")
def view_search(request: HttpRequest) -> HttpResponse:
    return view_documents_mixed(request, "orgapy/search.html")


@permission_required("orgapy.view_category")
def view_categories(request: HttpRequest) -> HttpResponse:
    uncategorized = Note.objects.filter(user=request.user, categories__isnull=True).count()\
        + Sheet.objects.filter(user=request.user, categories__isnull=True).count()\
        + Map.objects.filter(user=request.user, categories__isnull=True).count()
    projects = Note.objects.filter(user=request.user, project__isnull=False).count()
    return render(request, "orgapy/categories.html", {
        "categories": Category.objects.filter(user=request.user),
        "specials": {
            "uncategorized": uncategorized,
            "projects": projects,
        },
    })


@permission_required("orgapy.view_category")
def view_category(request: HttpRequest, name: str) -> HttpResponse:
    if name in ["journal", "org"]:
        return render(request, f"orgapy/specials/journal.html", {"category": name})
    if name == "quote":
        return render(request, f"orgapy/specials/quote.html", {})
    if name == "all":
        return render(request, f"orgapy/specials/all.html", {})
    category = "uncategorized"
    if name == "projects":
        category = "projects"
    elif name != "uncategorized":
        category = find_user_object(Category, "name", name, request.user)
    return view_documents_mixed(request, "orgapy/category.html", category)


@permission_required("orgapy.change_category")
def view_edit_category(request: HttpRequest, object_id: str) -> HttpResponse:
    query = Category.objects.filter(id=object_id)
    if query.exists():
        category = query.get()
        if category.user == request.user:
            if request.method == "POST":
                new_name = request.POST.get("name")
                if new_name is None:
                    raise BadRequest()
                if len(new_name) > 0:
                    category.name = new_name.lower()
                    category.save()
            return render(request, "orgapy/edit_category.html", {
                "category": category,
            })
    return redirect("orgapy:categories")


@permission_required("orgapy.delete_category")
def view_delete_category(request: HttpRequest, object_id: str) -> HttpResponse:
    query = Category.objects.filter(id=object_id)
    if query.exists():
        category = query.get()
        if category.user == request.user:
            category.delete()
    return redirect("orgapy:categories")


@permission_required("orgapy.change_note")
@permission_required("orgapy.change_sheet")
@permission_required("orgapy.change_map")
def view_edit(request: HttpRequest, active: str, object_id: str) -> HttpResponse:
    if active == "notes":
        return view_edit_note(request, object_id)
    if active == "sheets":
        return view_edit_sheet(request, object_id)
    if active == "maps":
        return view_edit_map(request, object_id)
    raise BadRequest(f"Unknown environment '{active}'")


@permission_required("orgapy.change_note")
@permission_required("orgapy.change_sheet")
@permission_required("orgapy.change_map")
def view_toggle_pin(request: HttpRequest, active: str, object_id: str) -> HttpResponse:
    return toggle_object_attribute(request, active, object_id, "pinned")


@permission_required("orgapy.change_note")
@permission_required("orgapy.change_sheet")
@permission_required("orgapy.change_map")
def view_toggle_public(request: HttpRequest, active: str, object_id: str) -> HttpResponse:
    return toggle_object_attribute(request, active, object_id, "public")


@permission_required("orgapy.view_note")
@permission_required("orgapy.view_sheet")
@permission_required("orgapy.view_map")
def view_export(request: HttpRequest, active: str, object_id: str) -> HttpResponse:
    if active == "notes":
        return view_export_note(request, object_id)
    if active == "sheets":
        return view_export_sheet(request, object_id)
    if active == "maps":
        return view_export_map(request, object_id)
    raise BadRequest(f"Unknown environment '{active}'")


@permission_required("orgapy.delete_note")
@permission_required("orgapy.delete_sheet")
@permission_required("orgapy.delete_map")
def view_delete(request: HttpRequest, active: str, object_id: str) -> HttpResponse:
    if active == "notes":
        return view_delete_note(request, object_id)
    if active == "sheets":
        return view_delete_sheet(request, object_id)
    if active == "maps":
        return view_delete_map(request, object_id)
    elif active == "subscription":
        return view_delete_subscription(request, object_id)
    raise BadRequest(f"Unknown environment '{active}'")


def view_share(request: HttpRequest, active: str, nonce: str) -> HttpResponse:
    if active == "notes":
        return view_note(request, nonce)
    if active == "sheets":
        return view_sheet(request, nonce)
    if active == "maps":
        return view_map(request, nonce)
    raise BadRequest(f"Unknown environment '{active}'")


@permission_required("orgapy.delete_note")
@permission_required("orgapy.delete_sheet")
@permission_required("orgapy.delete_map")
def view_restore(request: HttpRequest, active: str, object_id: str) -> HttpResponse:
    try:
        model = {"notes": Note, "sheets": Sheet, "maps": Map}[active]
    except KeyError:
        raise BadRequest()
    obj: Note | Sheet | Map = find_user_object(model, "id", object_id, request.user, allow_deleted=True)
    obj.restore()
    if "next" in request.GET:
        return redirect(request.GET["next"])
    return redirect(obj.get_absolute_url())


@permission_required("orgapy.delete_note")
@permission_required("orgapy.delete_sheet")
@permission_required("orgapy.delete_map")
def view_destroy(request: HttpRequest, active: str, object_id: str) -> HttpResponse:    
    try:
        model = {"notes": Note, "sheets": Sheet, "maps": Map}[active]
    except KeyError:
        raise BadRequest()
    obj: Note | Sheet | Map = find_user_object(model, "id", object_id, request.user, allow_deleted=True)
    obj.delete()
    if "next" in request.GET:
        return redirect(request.GET["next"])
    return redirect(f"orgapy:{active}")


@permission_required("orgapy.delete_note")
@permission_required("orgapy.delete_sheet")
@permission_required("orgapy.delete_map")
def view_trash(request: HttpRequest) -> HttpResponse:
    return view_documents_mixed(request, "orgapy/trash.html", None, deleted=True)


@permission_required("orgapy.delete_note")
@permission_required("orgapy.delete_sheet")
@permission_required("orgapy.delete_map")
def view_restore_all(request: HttpRequest) -> HttpResponse:
    Note.objects.filter(user=request.user, deleted=True).update(deleted=False, date_deletion=None)
    Sheet.objects.filter(user=request.user, deleted=True).update(deleted=False, date_deletion=None)
    Map.objects.filter(user=request.user, deleted=True).update(deleted=False, date_deletion=None)
    return redirect(f"orgapy:trash")


@permission_required("orgapy.delete_note")
@permission_required("orgapy.delete_sheet")
@permission_required("orgapy.delete_map")
def view_destroy_all(request: HttpRequest) -> HttpResponse:
    Note.objects.filter(user=request.user, deleted=True).delete()
    Sheet.objects.filter(user=request.user, deleted=True).delete()
    Map.objects.filter(user=request.user, deleted=True).delete()
    return redirect(f"orgapy:trash")


# NOTES ########################################################################


@permission_required("orgapy.view_note")
def view_notes(request: HttpRequest) -> HttpResponse:
    return view_documents_single(request, Note, "orgapy/notes.html", "notes")


@permission_required("orgapy.add_note")
def view_create_note(request: HttpRequest) -> HttpResponse:
    categories = Category.objects.filter(user=request.user)
    return render(request, "orgapy/create_note.html", {
        "categories": categories,
        "note_category_ids": {},
        "active": "notes",
    })


@permission_required("orgapy.change_note")
def view_save_note(request: HttpRequest) -> HttpResponse:
    if request.method == "POST":
        try:
            note = save_document_core(request, Note, ["content"])
        except ConflictError:
            return HttpResponse(content="Newer changes were made", content_type="text/plain", status=409)
        return redirect("orgapy:note", object_id=note.id)
    raise BadRequest()


def view_note(request: HttpRequest, object_id: str) -> HttpResponse:
    note = find_user_object(Note, ["id", "nonce"], object_id)
    has_permission = False
    readonly = True
    if request.user is not None and note.user == request.user and request.user.has_perm("orgapy.view_note"):
        readonly =  False
        has_permission = True
    elif note.public and isinstance(object_id, str) and len(object_id) == 12:
        has_permission = True
    if not has_permission:
        raise PermissionDenied()
    return render(request, "orgapy/note.html", {
        "note": note,
        "readonly": readonly,
        "active": "notes",
    })


def view_note_standalone(request: HttpRequest, object_id: str) -> HttpResponse:
    note = find_user_object(Note, ["id", "nonce"], object_id)
    has_permission = False
    if request.user is not None and note.user == request.user and request.user.has_perm("orgapy.view_note"):
        has_permission = True
    elif note.public and isinstance(object_id, str) and len(object_id) == 12:
        has_permission = True
    if not has_permission:
        raise PermissionDenied()
    response = render(request, "orgapy/note_standalone.html", {
        "note": note,
        "readonly": True,
    })
    response["X-Frame-Options"] = "SAMEORIGIN"
    return response


@permission_required("orgapy.change_note")
def view_edit_note(request: HttpRequest, object_id: str) -> HttpResponse:
    note = find_user_object(Note, "id", object_id, request.user)
    categories = Category.objects.filter(user=request.user).order_by("name")
    selected_category_ids = [category.id for category in note.categories.all()]
    return render(request, "orgapy/edit_note.html", {
        "note": note,
        "categories": categories,
        "selected_category_ids": selected_category_ids,
        "active": "notes",
    })


@permission_required("orgapy.view_note")
def view_export_note(request: HttpRequest, object_id: str) -> HttpResponse:
    """View to export a note's content as Markdown"""
    note = find_user_object(Note, "id", object_id, request.user)
    markdown = note.title + "\n\n" + note.content
    response = HttpResponse(content=markdown, content_type="text/markdown")
    response["Content-Disposition"] = "inline; filename=\"{}.md\"".format(slugify(note.title))
    return response


@permission_required("orgapy.delete_note")
def view_delete_note(request: HttpRequest, object_id: str) -> HttpResponse:
    """View to delete a note"""
    note = find_user_object(Note, "id", object_id, request.user)
    note.soft_delete()
    if "next" in request.GET:
        return redirect(request.GET["next"])
    return redirect("orgapy:notes")


@permission_required("orgapy.change_note")
def view_toggle_note_pin(request: HttpRequest, object_id: str) -> HttpResponse:
    return view_toggle_pin(request, "notes", object_id)


@permission_required("orgapy.change_note")
def view_toggle_note_public(request: HttpRequest, object_id: str) -> HttpResponse:
    return view_toggle_public(request, "notes", object_id)


@permission_required("orgapy.delete_note")
def view_restore_note(request, object_id: str) -> HttpResponse:
    return view_restore(request, "notes", object_id)


@permission_required("orgapy.delete_note")
def view_destroy_note(request, object_id: str) -> HttpResponse:
    return view_destroy(request, "notes", object_id)


@permission_required("orgapy.add_note")
def view_notally(request: HttpRequest) -> HttpResponse:
    created_notes = None
    if request.method == "POST":
        data = json.loads(request.POST.get("data", "[]"))
        notally_cat_query = Category.objects.filter(user=request.user, name="notally")
        notally_cat = None
        if notally_cat_query.exists():
            notally_cat = notally_cat_query.get()
        else:
            notally_cat = Category.objects.create(user=request.user, name="notally")
        created_notes = []
        for note_data in data:
            title = note_data["title"].strip()
            if not title:
                title = "Untitled Notally Note"
            note = Note.objects.create(
                user=request.user,
                title=title,
                content=note_data["content"],
            )
            note.date_creation = datetime.datetime.fromtimestamp(note_data["dateCreation"])
            note.save()
            note.categories.add(notally_cat)
            created_notes.append(note)
    return render(request, "orgapy/notally.html", {
        "notes": created_notes,
        "active": "notes",
    })


# SHEETS #######################################################################


@permission_required("orgapy.view_sheet")
def view_sheets(request: HttpRequest) -> HttpResponse:
    return view_documents_single(request, Sheet, "orgapy/sheets.html", "sheets")


@permission_required("orgapy.add_sheet")
def view_create_sheet(request: HttpRequest) -> HttpResponse:
    return render(request, "orgapy/create_sheet.html", {
        "active": "sheets",
    })


@permission_required("orgapy.change_sheet")
def view_save_sheet(request: HttpRequest) -> HttpResponse:
    if request.method == "POST":
        sheet = save_document_core(request, Sheet, ["description"])
        if "next" in request.POST:
            return redirect(request.POST["next"])
        return redirect("orgapy:sheet", object_id=sheet.id)
    raise BadRequest()


def view_sheet(request: HttpRequest, object_id: str) -> HttpResponse:
    sheet = find_user_object(Sheet, ["id", "nonce"], object_id)
    has_permission = False
    read_only = False
    if request.GET.get("embed"):
        read_only = True
    if request.user is not None and sheet.user == request.user and request.user.has_perm("orgapy.view_sheet"):
        has_permission = True
    elif sheet.public and isinstance(object_id, str) and len(object_id) == 12:
        has_permission = True
        read_only = True
    if not has_permission:
        raise PermissionDenied()
    response = render(request, "orgapy/sheet.html", {
        "sheet": sheet,
        "readonly": read_only,
        "active": "sheets",
    })
    response["X-Frame-Options"] = "SAMEORIGIN"
    return response


@permission_required("orgapy.change_sheet")
def view_edit_sheet(request: HttpRequest, object_id: str) -> HttpResponse:
    sheet = find_user_object(Sheet, "id", object_id, request.user)
    categories = Category.objects.filter(user=request.user).order_by("name")
    selected_category_ids = [category.id for category in sheet.categories.all()]
    return render(request, "orgapy/edit_sheet.html", {
        "sheet": sheet,
        "categories": categories,
        "selected_category_ids": selected_category_ids,
        "active": "sheets",
    })


@permission_required("orgapy.view_sheet")
def view_export_sheet(request: HttpRequest, object_id: str) -> HttpResponse:
    sheet = find_user_object(Sheet, ["id", "nonce"], object_id)
    if request.user is not None and sheet.user == request.user and request.user.has_perm("orgapy.view_sheet") or sheet.public:
        response = HttpResponse(sheet.data, content_type="text/tab-separated-values")
        response['Content-Disposition'] = f'attachment; filename="{sheet.title}.tsv"'
        return response
    raise PermissionDenied()


@permission_required("orgapy.delete_sheet")
def view_delete_sheet(request: HttpRequest, object_id: str) -> HttpResponse:
    sheet = find_user_object(Sheet, "id", object_id, request.user)
    sheet.soft_delete()
    if "next" in request.GET:
        return redirect(request.GET["next"])
    return redirect("orgapy:sheets")


@permission_required("orgapy.change_sheet")
def view_toggle_sheet_pin(request: HttpRequest, object_id: str) -> HttpResponse:
    return view_toggle_pin(request, "sheets", object_id)


@permission_required("orgapy.change_sheet")
def view_toggle_sheet_public(request: HttpRequest, object_id: str) -> HttpResponse:
    return view_toggle_public(request, "sheets", object_id)


@permission_required("orgapy.delete_sheet")
def view_restore_sheet(request, object_id: str) -> HttpResponse:
    return view_restore(request, "sheets", object_id)


@permission_required("orgapy.delete_sheet")
def view_destroy_sheet(request, object_id: str) -> HttpResponse:
    return view_destroy(request, "sheets", object_id)


# MAPS #########################################################################


@permission_required("orgapy.view_map")
def view_maps(request: HttpRequest) -> HttpResponse:
    return view_documents_single(request, Map, "orgapy/maps.html", "maps")


@permission_required("orgapy.add_map")
def view_create_map(request: HttpRequest) -> HttpResponse:
    return render(request, "orgapy/create_map.html", {
        "active": "maps",
    })


@permission_required("orgapy.change_map")
def view_edit_map(request: HttpRequest, object_id: str) -> HttpResponse:
    mmap = find_user_object(Map, "id", object_id, request.user)
    categories = Category.objects.filter(user=request.user).order_by("name")
    selected_category_ids = [category.id for category in mmap.categories.all()]
    return render(request, "orgapy/edit_map.html", {
        "map": mmap,
        "categories": categories,
        "selected_category_ids": selected_category_ids,
        "active": "maps",
    })


@permission_required("orgapy.change_map")
def view_save_map(request: HttpRequest) -> HttpResponse:
    if request.method == "POST":
        mmap = save_document_core(request, Map)
        if "next" in request.POST:
            return redirect(request.POST["next"])
        return redirect("orgapy:map", object_id=mmap.id)
    raise BadRequest()


def view_map(request: HttpRequest, object_id: str) -> HttpResponse:
    mmap = find_user_object(Map, ["id", "nonce"], object_id)
    has_permission = False
    read_only = True
    if not request.GET.get("embed"):
        read_only = False
    if request.user is not None and mmap.user == request.user and request.user.has_perm("orgapy.view_map"):
        has_permission = True
    elif mmap.public and isinstance(object_id, str) and len(object_id) == 12:
        has_permission = True
        read_only = True
    if not has_permission:
        raise PermissionDenied()
    response = render(request, "orgapy/map.html", {
        "map": mmap,
        "readonly": read_only,
        "active": "maps",
    })
    response["X-Frame-Options"] = "SAMEORIGIN"
    return response


@permission_required("orgapy.view_map")
def view_export_map(request: HttpRequest, object_id: str) -> HttpResponse:
    mmap = find_user_object(Map, ["id", "nonce"], object_id)
    if request.user is not None and mmap.user == request.user and request.user.has_perm("orgapy.view_map") or mmap.public:
        response = HttpResponse(mmap.geojson, content_type="application/geo+json")
        response['Content-Disposition'] = f'attachment; filename="{mmap.title}.geojson"'
        return response
    raise PermissionDenied()


@permission_required("orgapy.delete_map")
def view_delete_map(request: HttpRequest, object_id: str) -> HttpResponse:
    mmap = find_user_object(Map, "id", object_id, request.user)
    mmap.soft_delete()
    if "next" in request.GET:
        return redirect(request.GET["next"])
    return redirect("orgapy:maps")


@permission_required("orgapy.change_map")
def view_toggle_map_pin(request: HttpRequest, object_id: str) -> HttpResponse:
    return view_toggle_pin(request, "maps", object_id)


@permission_required("orgapy.change_map")
def view_toggle_map_public(request: HttpRequest, object_id: str) -> HttpResponse:
    return view_toggle_public(request, "maps", object_id)


@permission_required("orgapy.delete_map")
def view_restore_map(request, object_id: str) -> HttpResponse:
    return view_restore(request, "maps", object_id)


@permission_required("orgapy.delete_map")
def view_destroy_map(request, object_id: str) -> HttpResponse:
    return view_destroy(request, "maps", object_id)


# PROGRESS #####################################################################


@permission_required("orgapy.view_progress_log")
def view_progress(request: HttpRequest, year: int | str | None = None) -> HttpResponse:
    if year is None:
        year = datetime.datetime.now().year
    else:
        year = int(year)

    dt_start = datetime.datetime(year, 1, 1, 0, 0, 0, 0)
    dt_end = datetime.datetime(year, 12, 31, 0, 0, 0, 0)
    dt_filter = False
    try:
        if "start" in request.GET:
            dt_filter = True
            dt_start = datetime.datetime.strptime(request.GET["start"], "%Y-%m-%d")
        if "end" in request.GET:
            dt_filter = True
            dt_end = datetime.datetime.strptime(request.GET["end"], "%Y-%m-%d")
        if "date" in request.GET:
            dt_filter = True
            dt = datetime.datetime.strptime(request.GET["date"], "%Y-%m-%d")
            dt_start = dt
            dt_end = dt
    except ValueError:
        raise BadRequest()

    page_size = 24
    objects = ProgressLog.objects.filter(user=request.user).filter(dt__range=[dt_start, dt_end + datetime.timedelta(days=1)]).order_by("-dt")
    paginator = Paginator(objects, page_size)
    page = request.GET.get("page")
    logs = paginator.get_page(page)

    counter_query = ProgressCounter.objects.filter(user=request.user, year=year)
    counter = None
    if counter_query.exists():
        counter = counter_query.get()

    return render(request, "orgapy/progress.html", {
        "logs": logs,
        "year": year,
        "paginator": pretty_paginator(logs),
        "counter": counter,
        "dt_filter": dt_filter,
        "dt_start": dt_start,
        "dt_end": dt_end,
    })


@permission_required("orgapy.change_progress")
def view_progress_compute(request: HttpRequest, year: int | str | None = None) -> HttpResponse:
    if year is None:
        year = datetime.datetime.now().year
    else:
        year = int(year)
    counter_query = ProgressCounter.objects.filter(user=request.user, year=year)
    if not counter_query.exists():
        raise Http404()
    counter = counter_query.get()
    counter.recompute()
    return redirect("orgapy:progress_year", year=counter.year)


@permission_required("orgapy.view_progress")
def view_progress_export(request: HttpRequest, year: int | str | None = None) -> HttpResponse:
    if year is None:
        year = datetime.datetime.now().year
    else:
        year = int(year)
    lines = ["id\ttype\tdt\tdescription"]
    for log in ProgressLog.objects.filter(user=request.user, dt__year=year):
        lines.append("\t".join([
            str(log.id),
            log.type,
            log.dt.isoformat(),
            str(log.description)
        ]))
    return HttpResponse("\n".join(lines), content_type="text/tab-separated-values")


@permission_required("orgapy.add_progress_log")
def view_create_progress_log(request: HttpRequest) -> HttpResponse:
    return render(request, "orgapy/create_progress_log.html", {})


@permission_required("orgapy.change_progress_log")
def view_edit_progress_log(request: HttpRequest, object_id: str) -> HttpResponse:
    log = find_user_object(ProgressLog, "id", object_id, request.user)
    return render(request, "orgapy/edit_progress_log.html", {
        "log": log,
    })


@permission_required("orgapy.delete_progress_log")
def view_delete_progress_log(request: HttpRequest, object_id: str) -> HttpResponse:
    log = find_user_object(ProgressLog, "id", object_id, request.user)
    log.delete()
    return redirect("orgapy:progress")


@permission_required("orgapy.view_save_progress_log")
def view_save_progress_log(request: HttpRequest) -> HttpResponse:
    if request.method != "POST":
        raise BadRequest()
    original_log = None
    if ("id" in request.POST
        and ProgressLog.objects.filter(id=request.POST["id"]).exists()):
        original_log = ProgressLog.objects.get(id=request.POST["id"])
    if original_log is not None and original_log.user != request.user:
        raise PermissionDenied()
    dt_string = request.POST.get("dt")
    if dt_string is None:
        raise BadRequest()
    dt = datetime.datetime.strptime(dt_string, f"%Y-%m-%dT%H:%M")
    log_type = request.POST.get("type", ProgressLog.OTHER)
    description = request.POST.get("description")
    if original_log is None:
        ProgressLog.objects.create(
            user=request.user,
            dt=dt,
            type=log_type,
            description=description
        )
    else:
        original_log.dt = dt
        original_log.type = log_type
        original_log.description = description
        original_log.save()
    return redirect("orgapy:progress")


# SETTINGS #####################################################################


@permission_required("orgapy.change_settings")
def view_settings(request: HttpRequest) -> HttpResponse:
    if isinstance(request.user, AnonymousUser):
        raise PermissionDenied()
    user_settings = get_or_create_settings(request.user)
    if request.method == "POST":
        user_settings.objective_start_hours = int(request.POST.get("objective_start_hours", 0))
        user_settings.calendar_lookahead = int(request.POST.get("calendar_lookahead", 3))
        user_settings.trash_period = int(request.POST.get("trash_period", 30))
        user_settings.mood_log_hours = int(request.POST.get("mood_log_hours", 19))
        user_settings.mood_activities = request.POST.get("mood_activities", "").strip()
        user_settings.beach_mode = bool(request.POST.get("beach_mode", False))
        user_settings.save()
        if "ref" in request.POST and request.POST["ref"]:
            return redirect(request.POST["ref"])
    calendars = Calendar.objects.filter(user=request.user).order_by("calendar_name")
    return render(request, "orgapy/settings.html", {
        "settings": user_settings,
        "calendars": calendars,
        "subscriptions": PushSubscription.objects.filter(user=request.user),
        "VAPID_PUBLIC_KEY": settings.VAPID_PUBLIC_KEY,
    })


@permission_required("orgapy.change_calendar")
def view_calendar_form(request: HttpRequest) -> HttpResponse:
    if not request.method == "POST":
        raise BadRequest()
    if "id" in request.POST:
        query = Calendar.objects.filter(user=request.user, id=request.POST["id"])
        if not query.exists():
            raise Http404()
        calendar = query.get()
        if "save" in request.POST:
            calendar.url = request.POST["url"]
            calendar.username = request.POST["username"]
            calendar.password = request.POST["password"]
            calendar.calendar_name = request.POST["name"]
            calendar.sync_period = int(request.POST["sync_period"])
            calendar.save()
        elif "delete" in request.POST:
            calendar.delete()
        else:
            raise BadRequest()
    else:
        calendar = Calendar.objects.create(
            user=request.user,
            url=request.POST["url"],
            username=request.POST["username"],
            password=request.POST["password"],
            calendar_name=request.POST["name"],
            sync_period=int(request.POST["sync_period"]),
        )
    return redirect("orgapy:settings")


@permission_required("orgapy.delete_push_subscription")
def view_delete_subscription(request: HttpRequest, object_id: str) -> HttpResponse:
    sub = find_user_object(PushSubscription, "id", object_id, request.user)
    sub.delete()
    return redirect("orgapy:settings")


# MOOD #########################################################################


@permission_required("orgapy.view_mood_log")
def view_mood(request: HttpRequest) -> HttpResponse:
    if isinstance(request.user, AnonymousUser):
        raise PermissionDenied()
    settings = get_or_create_settings(request.user)
    pending_mood_logs = get_pending_mood_logs(request.user, settings.mood_log_hours)
    logs = MoodLog.objects.filter(user=request.user).order_by("-date")
    return render(request, "orgapy/mood.html", {
        "settings": settings,
        "pending_mood_logs": pending_mood_logs,
        "logs": logs,
    })


@permission_required("orgapy.delete_mood_log")
def view_delete_mood_log(request: HttpRequest, object_id: str) -> HttpResponse:
    mood_log = find_user_object(MoodLog, "id", object_id, request.user)
    mood_log.delete()
    if "next" in request.GET:
        return redirect(request.GET["next"])
    return redirect("orgapy:mood")
